/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Threading;

using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Persistence.Connection;

namespace Borland.Eco.Persistence
{
	public class ConnectionPool
	{
		private int m_MaxOpenConnections = 1;
		private int m_MaxPoolConnections = 1;
		private readonly Queue m_Pool = new Queue();
		private readonly ArrayList m_UsedConnections = new ArrayList();
		private PersistenceMapperDb m_Owner;
		private int m_Waiters;

		public ConnectionPool(PersistenceMapperDb owner)
		{
			m_Owner = owner;
		}

		public int MaxOpenConnections
		{
			get { return m_MaxOpenConnections; }
			set { m_MaxOpenConnections = value; }
		}

		public int MaxPoolConnections
		{
			get { return m_MaxPoolConnections; }
			set { m_MaxPoolConnections = value; }
		}

		private int EffectiveMaxPoolConnections 
		{
			get
			{
#if !CF
				if (m_Owner.Site != null)
					return 0;
				else
#endif
					return MaxPoolConnections;
			}
		}

		///<exception cref="InvalidOperationException">Thrown if <paramref name="exclusive"/> is true and there already are used connections.</exception>
		///<exception cref="InvalidOperationException">Thrown if the retrieved connection is already used.</exception>
		public IDatabase RetrieveClosedDatabaseConnection(bool exclusive)
		{
			lock(this)
			{
				IDatabase result = null;
				if (exclusive && (m_UsedConnections.Count > 0))
					throw new InvalidOperationException(PersistenceStringRes.sDatabaseAlreadyBusy);

				while (result == null)
				{
					if (m_Pool.Count > 0)
						result = (IDatabase)m_Pool.Dequeue();
					else if (m_UsedConnections.Count < MaxOpenConnections)
						result =  m_Owner.MakeDatabase();
					else
					{
						m_Waiters++;
#if !CF
						Monitor.Wait(this);
#endif
						m_Waiters--;
					}
				}
				if (m_UsedConnections.Contains(result))
					throw new InvalidOperationException("Internal error, connection already in use"); // Do not localize
				m_UsedConnections.Add(result);
				return result;
			}
		}

		public IDatabase RetrieveDatabaseConnection(bool exclusive)
		{
			IDatabase result = RetrieveClosedDatabaseConnection(exclusive);

			if (!result.Connected)
			{
				try
				{
					result.Open();
				}
				catch
				{
					ReturnDatabaseConnection(result);
					throw;
				}
			}
			return result;
		}

		///<exception cref="ArgumentNullException">Thrown if database is null.</exception>
		///<exception cref="ArgumentException">Thrown if database not not a used connection.</exception>
		public void ReturnDatabaseConnection(IDatabase database)
		{
			if (database == null) 
				throw new ArgumentNullException("database");
			lock(this)
			{
				int index = m_UsedConnections.IndexOf(database);
				if (index == -1)
					throw new ArgumentException(PersistenceStringRes.sDatabaseNotFromPool);
				m_UsedConnections.RemoveAt(index);

				if (database.InTransaction)
				{
					m_Owner.ReturnDatabase(database);
					throw new ArgumentException(PersistenceStringRes.sTransactionNotFinished);
				}

				if (m_Pool.Count < EffectiveMaxPoolConnections)
					m_Pool.Enqueue(database);
				else
					m_Owner.ReturnDatabase(database);

#if !CF
				if (m_Waiters > 0)
					Monitor.Pulse(this);
#endif
			}
		}

		public void CloseConnections(bool force)
		{
			lock(this)
			{
				while (m_Pool.Count > 0)
					m_Owner.ReturnDatabase((IDatabase)m_Pool.Dequeue());
				if (force)
				{
					foreach (IDatabase db in m_UsedConnections)
						m_Owner.ReturnDatabase(db);
					m_UsedConnections.Clear();
				}
			}
		}
	}
}
